<?php
header('Content-Type: application/json');
require '../config/config.php'; // Conexión a MySQL optimizada

// Configurar conexión persistente con tiempo de espera
$conexion->options(MYSQLI_OPT_CONNECT_TIMEOUT, 10);
session_start(); // Iniciar sesión si aún no está iniciada

// Verificar si el usuario está autenticado
if (!isset($_SESSION['usuario_id'])) {
    http_response_code(401); // No autorizado
    echo json_encode(['error' => 'No autorizado']);
    exit;
}

// Obtener y Sanitizar Datos
$cursoId = isset($_POST['searchInputID']) ? intval($_POST['searchInputID']) : 0;
$titulo = isset($_POST['nombreTitulostext']) ? trim($_POST['nombreTitulostext']) : '';
$descripcion = isset($_POST['descripciontext']) ? trim($_POST['descripciontext']) : '';

// Validaciones Backend
$errores = [];

if (empty($cursoId) || !filter_var($cursoId, FILTER_VALIDATE_INT)) {
    $errores['curso_id'] = 'El ID del curso es obligatorio y debe ser un número entero válido.';
} else {
    // Verificar si el curso existe en la base de datos
    $stmtCurso = $conexion->prepare("SELECT curso_id FROM cursos WHERE curso_id = ?");
    $stmtCurso->bind_param("i", $cursoId);
    $stmtCurso->execute();
    $resultCurso = $stmtCurso->get_result();
    if ($resultCurso->num_rows === 0) {
        $errores['curso_id'] = 'El curso seleccionado no existe.';
    }
    $stmtCurso->close();
}

// Validar título
if (empty($titulo)) {
    $errores['titulo'] = 'El título del cuestionario es obligatorio.';
} elseif (strlen($titulo) > 255) {
    $errores['titulo'] = 'El título no puede exceder los 255 caracteres.';
}

// Verificar si ya existe un cuestionario igual
if (empty($errores)) {
    $stmtExiste = $conexion->prepare("SELECT curso_id FROM cuestionarios WHERE curso_id = ? AND titulo = ?");
    $stmtExiste->bind_param("is", $cursoId, $titulo);
    $stmtExiste->execute();
    $resultExiste = $stmtExiste->get_result();

    if ($resultExiste->num_rows > 0) {
        $response = ['success' => false, 'message' => 'Ya existe un cuestionario con este título para el curso seleccionado.'];
        http_response_code(409); // Conflict
        echo json_encode($response);
        exit();
    }
    $stmtExiste->close();
}

// Si no hay errores, insertar en la base de datos
if (empty($errores)) {
    $stmtInsert = $conexion->prepare("INSERT INTO cuestionarios (curso_id, titulo, descripcion) VALUES (?, ?, ?)");
    $stmtInsert->bind_param("iss", $cursoId, $titulo, $descripcion);

    if ($stmtInsert->execute()) {
        $response = ['success' => true, 'message' => 'Cuestionario agregado correctamente.'];
        echo json_encode($response);
        exit();
    } else {
        // Log del error para depuración
        error_log("Error al guardar el cuestionario: " . $conexion->error);
        $response = ['success' => false, 'message' => 'Ocurrió un error al guardar el cuestionario en la base de datos.'];
        http_response_code(500); // Internal Server Error
        echo json_encode($response);
        exit();
    }
    $stmtInsert->close();
} else {
    // Si hay errores de validación
    $response = ['success' => false, 'message' => 'Error en los datos ingresados.', 'errors' => $errores];
    http_response_code(400); // Bad Request
    echo json_encode($response);
    exit();
}