<?php
require '../config/config.php';
header('Content-Type: application/json');

try {
    // Validar existencia de campos
    $requiredFields = ['nombre', 'rfc', 'curp', 'email', 'estado', 'promotoriaID', 'puestoID', 'rolID', 'estatus', 'fecha_ingreso', 'password'];

    foreach ($requiredFields as $field) {
        if (!isset($_POST[$field])) {
            throw new Exception("El campo '$field' no fue recibido.");
        }
        if (trim($_POST[$field]) === '') {
            throw new Exception("El campo '$field' no puede estar vacío.");
        }
    }

    // Validaciones específicas
    $email = filter_var(trim($_POST['email']), FILTER_VALIDATE_EMAIL);
    if (!$email) throw new Exception("El correo no es válido.");

    $rfc = strtoupper(trim($_POST['rfc']));
    if (!preg_match('/^[A-ZÑ&]{3,4}\d{6}[A-Z0-9]{3}$/', $rfc)) {
        throw new Exception("El RFC no tiene un formato válido.");
    }

    $curp = strtoupper(trim($_POST['curp']));
    if (!preg_match('/^[A-Z]{4}\d{6}[HM][A-Z]{5}[A-Z0-9]\d$/', $curp)) {
        throw new Exception("La CURP no tiene un formato válido.");
    }

    $fecha_ingreso = $_POST['fecha_ingreso'];
    if (!DateTime::createFromFormat('Y-m-d', $fecha_ingreso)) {
        throw new Exception("La fecha de ingreso no es válida.");
    }

    // Validar duplicados
    $stmt = $conexion->prepare("SELECT COUNT(*) FROM usuarios WHERE email = ? OR rfc = ? OR curp = ?");
    $stmt->bind_param("sss", $email, $rfc, $curp);
    $stmt->execute();
    $stmt->bind_result($count);
    $stmt->fetch();
    $stmt->close();

    if ($count > 0) {
        throw new Exception("Ya existe un usuario con el mismo RFC, CURP o correo.");
    }

    // Preparar datos para insertar
    $password_hash = password_hash($_POST['password'], PASSWORD_DEFAULT);


    // Guardar en la base de datos
    $stmt = $conexion->prepare("INSERT INTO usuarios 
        (nombre_completo, rfc, curp, email, password, estado_lugar, promotoria_id, puesto_id, rol_id, status, fecha_ingreso) 
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");

    $stmt->bind_param(
        "ssssssiiiss",
        $_POST['nombre'],      // nombre_completo
        $rfc,                  // rfc
        $curp,                 // curp
        $email,                // email
        $password_hash,        // password
        $_POST['estado'],      // estado_lugar
        $_POST['promotoriaID'], // promotoria_id
        $_POST['puestoID'],     // puesto_id
        $_POST['rolID'],        // rol_id
        $_POST['estatus'],      // status
        $fecha_ingreso         // fecha_ingreso
    );

    if ($stmt->execute()) {
        echo json_encode(['success' => true, 'message' => 'Usuario creado correctamente']);
    } else {
        throw new Exception("Error al guardar el usuario: " . $stmt->error);
    }

    $stmt->close();
    $conexion->close();
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}
