<?php
header('Content-Type: application/json');
require '../config/config.php'; // Ajusta la ruta según tu estructura

$response = [
    'success' => false,
    'data' => null,
    'error' => ''
];

try {
    // Validar parámetros recibidos
    $cursoId = isset($_GET['curso_id']) ? intval($_GET['curso_id']) : null;
    $tituloId = isset($_GET['titulo_id']) ? intval($_GET['titulo_id']) : null;
    $cuestionarioId = isset($_GET['cuestionario_id']) ? intval($_GET['cuestionario_id']) : null;

    if (!$cursoId || !$tituloId || !$cuestionarioId) {
        throw new Exception('Parámetros incompletos. Se requieren curso_id, titulo_id y cuestionario_id');
    }

    if ($conexion->connect_error) {
        throw new Exception("Error de conexión: " . $conexion->connect_error);
    }

    // Consulta principal para obtener los datos básicos
    $query = "
        SELECT 
            c.curso_id,
            c.curso_title AS nombre_curso,
            tt.ID_Titulo,
            tt.nom_titulo_temario AS nombre_titulo_temario,
            q.cuestionario_id,
            q.titulo AS nombre_cuestionario,
            q.descripcion AS descripcion_cuestionario,
            q.fecha_creacion
        FROM titulos_temarios_cuestionarios ttc
        JOIN titulos_temarios tt ON tt.ID_Titulo = ttc.ID_Titulo
        JOIN cuestionarios q ON q.cuestionario_id = ttc.cuestionario_id
        JOIN cursos c ON c.curso_id = tt.curso_id
        WHERE c.curso_id = ? 
          AND tt.ID_Titulo = ? 
          AND q.cuestionario_id = ?
        LIMIT 1
    ";

    $stmt = $conexion->prepare($query);
    $stmt->bind_param("iii", $cursoId, $tituloId, $cuestionarioId);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows === 0) {
        throw new Exception('No se encontró el cuestionario solicitado');
    }

    $row = $result->fetch_assoc();
    $stmt->close();

    // Obtener las preguntas y sus opciones
    $queryPreguntas = "
        SELECT 
            p.pregunta_id,
            p.texto_pregunta,
            p.puntaje
        FROM preguntas p
        WHERE p.cuestionario_id = ?
        ORDER BY p.pregunta_id
    ";

    $stmtPreguntas = $conexion->prepare($queryPreguntas);
    $stmtPreguntas->bind_param("i", $cuestionarioId);
    $stmtPreguntas->execute();
    $resultPreguntas = $stmtPreguntas->get_result();

    $preguntas = [];
    while ($pregunta = $resultPreguntas->fetch_assoc()) {
        // Obtener opciones para cada pregunta
        $queryOpciones = "
            SELECT 
                opcion_id,
                texto_opcion,
                es_correcta
            FROM opciones_pregunta
            WHERE pregunta_id = ?
            ORDER BY opcion_id
        ";

        $stmtOpciones = $conexion->prepare($queryOpciones);
        $stmtOpciones->bind_param("i", $pregunta['pregunta_id']);
        $stmtOpciones->execute();
        $resultOpciones = $stmtOpciones->get_result();

        $opciones = [];
        while ($opcion = $resultOpciones->fetch_assoc()) {
            $opciones[] = [
                'id' => $opcion['opcion_id'],
                'texto' => $opcion['texto_opcion'],
                'es_correcta' => (bool)$opcion['es_correcta'],
            ];
        }
        $stmtOpciones->close();

        $preguntas[] = [
            'id' => $pregunta['pregunta_id'],
            'texto' => $pregunta['texto_pregunta'],
            'puntaje' => floatval($pregunta['puntaje']),
            'opciones' => $opciones
        ];
    }
    $stmtPreguntas->close();

    // Calcular estadísticas
    $totalPreguntas = count($preguntas);
    $puntajeTotal = array_reduce($preguntas, function ($sum, $p) {
        return $sum + $p['puntaje'];
    }, 0);

    // Construir respuesta final
    $response['success'] = true;
    $response['data'] = [
        'curso' => [
            'id' => $row['curso_id'],
            'nombre' => $row['nombre_curso']
        ],
        'titulo_temario' => [
            'id' => $row['ID_Titulo'],
            'nombre' => $row['nombre_titulo_temario']
        ],
        'cuestionario' => [
            'id' => $row['cuestionario_id'],
            'nombre' => $row['nombre_cuestionario'],
            'descripcion' => $row['descripcion_cuestionario'],
            'fecha_creacion' => $row['fecha_creacion'],
            'total_preguntas' => $totalPreguntas,
            'puntaje_total' => $puntajeTotal
        ],
        'preguntas' => $preguntas,
        'estadisticas' => [
            'preguntas_sin_opciones' => count(array_filter($preguntas, function ($p) {
                return empty($p['opciones']);
            })),
            'preguntas_con_multiple_respuesta' => count(array_filter($preguntas, function ($p) {
                return count(array_filter($p['opciones'], function ($o) {
                    return $o['es_correcta'];
                })) > 1;
            }))
        ]
    ];
} catch (Exception $e) {
    $response['error'] = $e->getMessage();
    http_response_code(500); // Error interno del servidor
} finally {
    if (isset($conexion) && $conexion->ping()) {
        $conexion->close();
    }
}

echo json_encode($response);
