<?php
// Iniciar buffer de salida inmediatamente
ob_start();

// Configuración inicial
session_start();
require '../config/config.php';

// Establecer headers JSON
header('Content-Type: application/json; charset=utf-8');

function send_json_response($status, $message, $data = [])
{
    ob_end_clean();
    echo json_encode([
        'status' => $status,
        'message' => $message,
        'data' => $data
    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

try {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('Método no permitido', 405);
    }

    $required = [
        'searchInputID' => 'ID_Titulo',
        'nombrecontenidotext' => 'Nombre de contenido',
        'contentarea' => 'Descripción',
        'orden' => 'Nivel/Subtema'
    ];

    $missing = [];
    foreach ($required as $field => $name) {
        if (empty($_POST[$field])) {
            $missing[] = $name;
        }
    }

    if (!empty($missing)) {
        send_json_response('error', 'Campos obligatorios faltantes: ' . implode(', ', $missing), [
            'campos_faltantes' => $missing
        ]);
    }

    $ID_Titulo = $_POST['searchInputID'];
    $nombre_contenido = $_POST['nombrecontenidotext'];
    $descripcion_contenido = $_POST['contentarea'];
    $nivelSubtema = $_POST['orden'];

    $conexion->begin_transaction();

    $query = "INSERT INTO contenidos_titulos_temarios (ID_Titulo, nombre_contenido, descripcion_contenido, Orden)
              VALUES (?, ?, ?, ?)";
    $stmt = $conexion->prepare($query);
    if (!$stmt) {
        throw new Exception('Error al preparar consulta: ' . $conexion->error);
    }

    $stmt->bind_param("issi", $ID_Titulo, $nombre_contenido, $descripcion_contenido, $nivelSubtema);
    if (!$stmt->execute()) {
        throw new Exception('Error al insertar el contenido: ' . $conexion->error);
    }

    $contenido_id = $stmt->insert_id;

    $archivos_subidos = $_FILES['archivo_material'] ?? null;
    $respuesta_archivos = [];
    $upload_dir = 'uploads/materiales/';

    if (!file_exists($upload_dir) && !mkdir($upload_dir, 0755, true)) {
        throw new Exception('No se pudo crear el directorio de uploads');
    }

    if ($archivos_subidos && is_array($archivos_subidos['name'])) {
        $conexion->autocommit(false);
        $some_success = false;

        foreach ($archivos_subidos['name'] as $key => $file_name) {
            if (empty($file_name)) continue;

            $file_tmp = $archivos_subidos['tmp_name'][$key];
            $file_size = $archivos_subidos['size'][$key];
            $file_error = $archivos_subidos['error'][$key];

            if ($file_error !== UPLOAD_ERR_OK) {
                $respuesta_archivos[] = [
                    'file' => $file_name,
                    'status' => 'error',
                    'message' => uploadErrorMessages($file_error)
                ];
                continue;
            }

            $ext = strtolower(pathinfo($file_name, PATHINFO_EXTENSION));
            $mime_real = guessMimeTypeByExtension($ext);

            $allowed = [
                'pdf' => ['pdf'],
                'video' => ['mp4', 'mov', 'avi', 'mkv', 'webm'],
                'word' => ['doc', 'docx'],
                'image' => ['jpg', 'jpeg', 'png', 'gif']
            ];

            $tipo_material = 'Otro';
            $permitido = false;

            foreach ($allowed as $tipo => $exts) {
                if (in_array($ext, $exts)) {
                    $tipo_material = ucfirst($tipo);
                    $permitido = true;
                    break;
                }
            }

            $max_size = ($tipo_material === 'Video') ? 500000000 : 50000000;

            if (!$permitido) {
                $respuesta_archivos[] = [
                    'file' => $file_name,
                    'status' => 'error',
                    'message' => 'Tipo de archivo no permitido (' . $ext . ')'
                ];
                continue;
            }

            if ($file_size > $max_size) {
                $respuesta_archivos[] = [
                    'file' => $file_name,
                    'status' => 'error',
                    'message' => 'Archivo excede tamaño máximo (' . formatBytes($max_size) . ')'
                ];
                continue;
            }

            $new_name = 'file_' . md5(uniqid() . $file_name) . '.' . $ext;
            $destino = $upload_dir . $new_name;

            if (!@move_uploaded_file($file_tmp, $destino)) {
                $error = error_get_last();
                $respuesta_archivos[] = [
                    'file' => $file_name,
                    'status' => 'error',
                    'message' => 'Error al mover el archivo: ' . ($error['message'] ?? 'Error desconocido')
                ];
                continue;
            }

            $sql = "INSERT INTO materiales_contenido (id_Contenido, tipo_material, ruta_archivo) VALUES (?, ?, ?)";
            $stmt_material = $conexion->prepare($sql);
            $stmt_material->bind_param("iss", $contenido_id, $tipo_material, $destino);

            if ($stmt_material->execute()) {
                $respuesta_archivos[] = [
                    'file' => $file_name,
                    'status' => 'success',
                    'id_material' => $stmt_material->insert_id,
                    'tipo' => $tipo_material,
                    'size' => formatBytes($file_size),
                    'mime_type' => $mime_real
                ];
                $some_success = true;
            } else {
                @unlink($destino);
                $respuesta_archivos[] = [
                    'file' => $file_name,
                    'status' => 'error',
                    'message' => 'Error en base de datos: ' . $conexion->error
                ];
            }
        }

        if ($some_success) {
            $conexion->commit();
        } else {
            $conexion->rollback();
        }

        $conexion->autocommit(true);
    }

    $errors_in_files = array_filter($respuesta_archivos, fn($item) => $item['status'] === 'error');

    if (empty($respuesta_archivos)) {
        send_json_response('success', 'Contenido guardado correctamente (sin archivos adjuntos)', [
            'id_contenido' => $contenido_id
        ]);
    } elseif (empty($errors_in_files)) {
        send_json_response('success', 'Contenido y archivos guardados correctamente', [
            'id_contenido' => $contenido_id,
            'archivos' => $respuesta_archivos
        ]);
    } else {
        send_json_response('partial', 'Contenido guardado, pero con errores en algunos archivos', [
            'id_contenido' => $contenido_id,
            'archivos' => $respuesta_archivos
        ]);
    }

} catch (Exception $e) {
    if ($conexion && $conexion->autocommit === false) {
        $conexion->rollback();
    }

    send_json_response('error', $e->getMessage(), [
        'trace' => $e->getTraceAsString()
    ]);
}

function uploadErrorMessages($code)
{
    $errors = [
        UPLOAD_ERR_INI_SIZE => 'El archivo excede el tamaño permitido',
        UPLOAD_ERR_FORM_SIZE => 'El archivo excede el tamaño permitido por el formulario',
        UPLOAD_ERR_PARTIAL => 'El archivo solo se subió parcialmente',
        UPLOAD_ERR_NO_FILE => 'No se seleccionó ningún archivo',
        UPLOAD_ERR_NO_TMP_DIR => 'Error del servidor: no hay directorio temporal',
        UPLOAD_ERR_CANT_WRITE => 'Error del servidor: no se pudo guardar el archivo',
        UPLOAD_ERR_EXTENSION => 'Extensión de archivo no permitida'
    ];
    return $errors[$code] ?? 'Error desconocido al subir el archivo';
}

function formatBytes($bytes, $precision = 2)
{
    $units = ['B', 'KB', 'MB', 'GB'];
    $bytes = max($bytes, 0);
    $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
    $pow = min($pow, count($units) - 1);
    $bytes /= pow(1024, $pow);
    return round($bytes, $precision) . ' ' . $units[$pow];
}

function guessMimeTypeByExtension($ext)
{
    $mimes = [
        'pdf' => 'application/pdf',
        'mp4' => 'video/mp4',
        'mov' => 'video/quicktime',
        'avi' => 'video/x-msvideo',
        'mkv' => 'video/x-matroska',
        'webm' => 'video/webm',
        'doc' => 'application/msword',
        'docx' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
        'jpg' => 'image/jpeg',
        'jpeg' => 'image/jpeg',
        'png' => 'image/png',
        'gif' => 'image/gif'
    ];
    return $mimes[$ext] ?? 'application/octet-stream';
}
